/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.josm.io;

import jakarta.json.Json;
import jakarta.json.JsonArray;
import jakarta.json.JsonNumber;
import jakarta.json.JsonObject;
import jakarta.json.JsonString;
import jakarta.json.JsonValue;
import jakarta.json.stream.JsonParser;
import jakarta.json.stream.JsonParsingException;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import org.openstreetmap.josm.data.coor.EastNorth;
import org.openstreetmap.josm.data.coor.LatLon;
import org.openstreetmap.josm.data.osm.BBox;
import org.openstreetmap.josm.data.osm.DataSet;
import org.openstreetmap.josm.data.osm.IRelation;
import org.openstreetmap.josm.data.osm.Node;
import org.openstreetmap.josm.data.osm.OsmPrimitive;
import org.openstreetmap.josm.data.osm.Relation;
import org.openstreetmap.josm.data.osm.RelationMember;
import org.openstreetmap.josm.data.osm.Tag;
import org.openstreetmap.josm.data.osm.TagCollection;
import org.openstreetmap.josm.data.osm.TagMap;
import org.openstreetmap.josm.data.osm.UploadPolicy;
import org.openstreetmap.josm.data.osm.Way;
import org.openstreetmap.josm.data.projection.Projection;
import org.openstreetmap.josm.data.projection.Projections;
import org.openstreetmap.josm.data.validation.TestError;
import org.openstreetmap.josm.data.validation.tests.DuplicateWay;
import org.openstreetmap.josm.gui.conflict.tags.TagConflictResolutionUtil;
import org.openstreetmap.josm.gui.conflict.tags.TagConflictResolverModel;
import org.openstreetmap.josm.gui.progress.ProgressMonitor;
import org.openstreetmap.josm.io.AbstractReader;
import org.openstreetmap.josm.io.IllegalDataException;
import org.openstreetmap.josm.tools.CheckParameterUtil;
import org.openstreetmap.josm.tools.I18n;
import org.openstreetmap.josm.tools.Logging;
import org.openstreetmap.josm.tools.Utils;

public class GeoJSONReader
extends AbstractReader {
    private static final String CRS = "crs";
    private static final String NAME = "name";
    private static final String LINK = "link";
    private static final String COORDINATES = "coordinates";
    private static final String FEATURES = "features";
    private static final String PROPERTIES = "properties";
    private static final String GEOMETRY = "geometry";
    private static final String TYPE = "type";
    private static final byte RECORD_SEPARATOR_BYTE = 30;
    private static final String CRS_GEOJSON = "EPSG:4326";
    private Projection projection = Projections.getProjectionByCode("EPSG:4326");

    GeoJSONReader() {
    }

    private void parse(JsonParser parser) throws IllegalDataException {
        while (parser.hasNext()) {
            JsonParser.Event event = parser.next();
            if (event != JsonParser.Event.START_OBJECT) continue;
            this.parseRoot(parser.getObject());
        }
        parser.close();
    }

    private void parseRoot(JsonObject object) throws IllegalDataException {
        this.parseCrs(object.getJsonObject(CRS));
        switch (Optional.ofNullable(object.getJsonString(TYPE)).orElseThrow(() -> new IllegalDataException("No type")).getString()) {
            case "FeatureCollection": {
                JsonValue.ValueType valueType = ((JsonValue)object.get(FEATURES)).getValueType();
                CheckParameterUtil.ensureThat(valueType == JsonValue.ValueType.ARRAY, "features must be ARRAY, but is " + (Object)((Object)valueType));
                this.parseFeatureCollection(object.getJsonArray(FEATURES), false);
                break;
            }
            case "Feature": {
                this.parseFeature(object);
                break;
            }
            case "GeometryCollection": {
                this.parseGeometryCollection(null, object, false);
                break;
            }
            default: {
                this.parseGeometry(null, object);
            }
        }
    }

    private void parseCrs(JsonObject crs) throws IllegalDataException {
        JsonObject properties;
        if (crs != null && (properties = crs.getJsonObject(PROPERTIES)) != null) {
            switch (crs.getString(TYPE)) {
                case "name": {
                    String crsName = properties.getString(NAME);
                    if ("urn:ogc:def:crs:OGC:1.3:CRS84".equals(crsName)) {
                        crsName = CRS_GEOJSON;
                    } else if (crsName.startsWith("urn:ogc:def:crs:EPSG:")) {
                        crsName = crsName.replace("urn:ogc:def:crs:", "");
                    }
                    this.projection = Optional.ofNullable(Projections.getProjectionByCode(crsName)).orElse(Projections.getProjectionByCode(CRS_GEOJSON));
                    break;
                }
                default: {
                    throw new IllegalDataException(crs.toString());
                }
            }
        }
    }

    private Optional<? extends OsmPrimitive> parseFeatureCollection(JsonArray features, boolean createRelation) {
        List primitives = features.stream().filter(JsonObject.class::isInstance).map(JsonObject.class::cast).map(this::parseFeature).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
        if (createRelation && primitives.size() > 1) {
            Relation relation = new Relation();
            relation.setMembers(primitives.stream().map(osm -> new RelationMember("", (OsmPrimitive)osm)).collect(Collectors.toList()));
            this.getDataSet().addPrimitive(relation);
            return Optional.of(relation);
        }
        if (primitives.size() == 1) {
            return Optional.of((OsmPrimitive)primitives.get(0));
        }
        return Optional.empty();
    }

    private Optional<? extends OsmPrimitive> parseFeature(JsonObject feature) {
        JsonValue geometry = (JsonValue)feature.get(GEOMETRY);
        if (geometry != null && geometry.getValueType() == JsonValue.ValueType.OBJECT) {
            return this.parseGeometry(feature, geometry.asJsonObject());
        }
        JsonValue properties = (JsonValue)feature.get(PROPERTIES);
        if (properties != null && properties.getValueType() == JsonValue.ValueType.OBJECT) {
            return this.parseNonGeometryFeature(feature, properties.asJsonObject());
        }
        Logging.warn(I18n.tr("Relation/non-geometry feature without properties found: {0}", feature));
        return Optional.empty();
    }

    private Optional<? extends OsmPrimitive> parseNonGeometryFeature(JsonObject feature, JsonObject properties) {
        Optional<? extends OsmPrimitive> osm;
        JsonValue type = (JsonValue)properties.get(TYPE);
        if (type == null || properties.getValueType() == JsonValue.ValueType.STRING) {
            Logging.warn(I18n.tr("Relation/non-geometry feature without type found: {0}", feature));
            if (!feature.containsKey(FEATURES)) {
                return Optional.empty();
            }
        }
        OsmPrimitive primitive = null;
        if (feature.containsKey(FEATURES) && ((JsonValue)feature.get(FEATURES)).getValueType() == JsonValue.ValueType.ARRAY && (osm = this.parseFeatureCollection(feature.getJsonArray(FEATURES), true)).isPresent()) {
            primitive = osm.get();
            GeoJSONReader.fillTagsFromFeature(feature, primitive);
        }
        return Optional.ofNullable(primitive);
    }

    private Optional<Relation> parseGeometryCollection(JsonObject feature, JsonObject geometry, boolean createRelation) {
        ArrayList<RelationMember> relationMembers = new ArrayList<RelationMember>(geometry.getJsonArray("geometries").size());
        for (JsonValue jsonValue : geometry.getJsonArray("geometries")) {
            this.parseGeometry(feature, jsonValue.asJsonObject()).map(osm -> new RelationMember("", (OsmPrimitive)osm)).ifPresent(relationMembers::add);
        }
        if (createRelation) {
            Relation relation = new Relation();
            relation.setMembers((List<RelationMember>)relationMembers);
            this.getDataSet().addPrimitive(relation);
            return Optional.of(GeoJSONReader.fillTagsFromFeature(feature, relation));
        }
        return Optional.empty();
    }

    private Optional<? extends OsmPrimitive> parseGeometry(JsonObject feature, JsonObject geometry) {
        if (geometry == null) {
            GeoJSONReader.parseNullGeometry(feature);
            return Optional.empty();
        }
        switch (geometry.getString(TYPE)) {
            case "Point": {
                return this.parsePoint(feature, geometry.getJsonArray(COORDINATES));
            }
            case "MultiPoint": {
                return this.parseMultiPoint(feature, geometry);
            }
            case "LineString": {
                return this.parseLineString(feature, geometry.getJsonArray(COORDINATES));
            }
            case "MultiLineString": {
                return this.parseMultiLineString(feature, geometry);
            }
            case "Polygon": {
                return this.parsePolygon(feature, geometry.getJsonArray(COORDINATES));
            }
            case "MultiPolygon": {
                return this.parseMultiPolygon(feature, geometry);
            }
            case "GeometryCollection": {
                return this.parseGeometryCollection(feature, geometry, true);
            }
        }
        GeoJSONReader.parseUnknown(geometry);
        return Optional.empty();
    }

    private LatLon getLatLon(JsonArray coordinates) {
        return this.projection.eastNorth2latlon(new EastNorth(GeoJSONReader.parseCoordinate((JsonValue)coordinates.get(0)), GeoJSONReader.parseCoordinate((JsonValue)coordinates.get(1))));
    }

    private static double parseCoordinate(JsonValue coordinate) {
        if (coordinate instanceof JsonString) {
            return Double.parseDouble(((JsonString)coordinate).getString());
        }
        if (coordinate instanceof JsonNumber) {
            return ((JsonNumber)coordinate).doubleValue();
        }
        throw new IllegalArgumentException(Objects.toString(coordinate));
    }

    private Optional<Node> parsePoint(JsonObject feature, JsonArray coordinates) {
        return Optional.of(GeoJSONReader.fillTagsFromFeature(feature, this.createNode(this.getLatLon(coordinates))));
    }

    private Optional<Relation> parseMultiPoint(JsonObject feature, JsonObject geometry) {
        ArrayList<RelationMember> nodes = new ArrayList<RelationMember>(geometry.getJsonArray(COORDINATES).size());
        for (JsonValue coordinate : geometry.getJsonArray(COORDINATES)) {
            this.parsePoint(feature, coordinate.asJsonArray()).map(node -> new RelationMember("", (OsmPrimitive)node)).ifPresent(nodes::add);
        }
        Relation returnRelation = new Relation();
        returnRelation.setMembers((List<RelationMember>)nodes);
        this.getDataSet().addPrimitive(returnRelation);
        return Optional.of(GeoJSONReader.fillTagsFromFeature(feature, returnRelation));
    }

    private Optional<Way> parseLineString(JsonObject feature, JsonArray coordinates) {
        if (!coordinates.isEmpty()) {
            Optional<Way> way = this.createWay(coordinates, false);
            way.ifPresent(tWay -> GeoJSONReader.fillTagsFromFeature(feature, tWay));
            return way;
        }
        return Optional.empty();
    }

    private Optional<Relation> parseMultiLineString(JsonObject feature, JsonObject geometry) {
        ArrayList<RelationMember> ways = new ArrayList<RelationMember>(geometry.getJsonArray(COORDINATES).size());
        for (JsonValue coordinate : geometry.getJsonArray(COORDINATES)) {
            this.parseLineString(feature, coordinate.asJsonArray()).map(way -> new RelationMember("", (OsmPrimitive)way)).ifPresent(ways::add);
        }
        Relation relation = new Relation();
        relation.setMembers((List<RelationMember>)ways);
        this.getDataSet().addPrimitive(relation);
        return Optional.of(GeoJSONReader.fillTagsFromFeature(feature, relation));
    }

    private Optional<? extends OsmPrimitive> parsePolygon(JsonObject feature, JsonArray coordinates) {
        int size = coordinates.size();
        if (size == 1) {
            Optional<Way> optionalWay = this.createWay(coordinates.getJsonArray(0), true);
            optionalWay.ifPresent(way -> GeoJSONReader.fillTagsFromFeature(feature, way));
            return optionalWay;
        }
        if (size > 1) {
            Relation multipolygon = new Relation();
            this.createWay(coordinates.getJsonArray(0), true).ifPresent(way -> multipolygon.addMember(new RelationMember("outer", (OsmPrimitive)way)));
            for (JsonValue interiorRing : coordinates.subList(1, size)) {
                this.createWay(interiorRing.asJsonArray(), true).ifPresent(way -> multipolygon.addMember(new RelationMember("inner", (OsmPrimitive)way)));
            }
            GeoJSONReader.fillTagsFromFeature(feature, multipolygon);
            multipolygon.put(TYPE, "multipolygon");
            this.getDataSet().addPrimitive(multipolygon);
            return Optional.of(multipolygon);
        }
        return Optional.empty();
    }

    private Optional<Relation> parseMultiPolygon(JsonObject feature, JsonObject geometry) {
        ArrayList<RelationMember> relationMembers = new ArrayList<RelationMember>(geometry.getJsonArray(COORDINATES).size());
        for (JsonValue coordinate : geometry.getJsonArray(COORDINATES)) {
            this.parsePolygon(feature, coordinate.asJsonArray()).map(poly -> new RelationMember("", (OsmPrimitive)poly)).ifPresent(relationMembers::add);
        }
        Relation relation = new Relation();
        relation.setMembers((List<RelationMember>)relationMembers);
        return Optional.of(GeoJSONReader.fillTagsFromFeature(feature, relation));
    }

    private Node createNode(LatLon latlon) {
        List<Node> existingNodes = this.getDataSet().searchNodes(new BBox(latlon, latlon));
        if (!existingNodes.isEmpty()) {
            return existingNodes.get(0);
        }
        Node node = new Node(latlon);
        this.getDataSet().addPrimitive(node);
        return node;
    }

    private Optional<Way> createWay(JsonArray coordinates, boolean autoClose) {
        if (coordinates.isEmpty()) {
            return Optional.empty();
        }
        List latlons = coordinates.stream().map(coordinate -> this.getLatLon(coordinate.asJsonArray())).collect(Collectors.toList());
        int size = latlons.size();
        boolean doAutoclose = size > 1 ? (((LatLon)latlons.get(0)).equals(latlons.get(size - 1)) ? false : autoClose) : false;
        Way way = new Way();
        this.getDataSet().addPrimitive(way);
        List rawNodes = latlons.stream().map(this::createNode).collect(Collectors.toList());
        if (doAutoclose) {
            rawNodes.add((Node)rawNodes.get(0));
        }
        ArrayList<Node> wayNodes = new ArrayList<Node>(rawNodes.size());
        Node last = null;
        for (Node curr : rawNodes) {
            if (last != curr) {
                wayNodes.add(curr);
            }
            last = curr;
        }
        way.setNodes((List<Node>)wayNodes);
        return Optional.of(way);
    }

    private static <O extends OsmPrimitive> O fillTagsFromFeature(JsonObject feature, O primitive) {
        if (feature != null) {
            TagCollection featureTags = GeoJSONReader.getTags(feature);
            primitive.setKeys(new TagMap(primitive.isTagged() ? GeoJSONReader.mergeAllTagValues(primitive, featureTags) : featureTags));
        }
        return primitive;
    }

    private static TagCollection mergeAllTagValues(OsmPrimitive primitive, TagCollection featureTags) {
        TagCollection tags = TagCollection.from(primitive).union(featureTags);
        TagConflictResolutionUtil.applyAutomaticTagConflictResolution(tags);
        TagConflictResolutionUtil.normalizeTagCollectionBeforeEditing(tags, Collections.singletonList(primitive));
        TagConflictResolverModel tagModel = new TagConflictResolverModel();
        tagModel.populate(new TagCollection(tags), tags.getKeysWithMultipleValues());
        tagModel.actOnDecisions((k, d) -> d.keepAll());
        return tagModel.getAllResolutions();
    }

    private static void parseUnknown(JsonObject object) {
        Logging.warn(I18n.tr("Unknown json object found {0}", object));
    }

    private static void parseNullGeometry(JsonObject feature) {
        Logging.warn(I18n.tr("Geometry of feature {0} is null", feature));
    }

    private static TagCollection getTags(JsonObject feature) {
        JsonValue properties;
        TagCollection tags = new TagCollection();
        if (feature.containsKey(PROPERTIES) && !feature.isNull(PROPERTIES) && (properties = (JsonValue)feature.get(PROPERTIES)) != null && properties.getValueType() == JsonValue.ValueType.OBJECT) {
            for (Map.Entry stringJsonValueEntry : properties.asJsonObject().entrySet()) {
                JsonValue value = (JsonValue)stringJsonValueEntry.getValue();
                if (value instanceof JsonString) {
                    tags.add(new Tag((String)stringJsonValueEntry.getKey(), ((JsonString)value).getString()));
                    continue;
                }
                if (value instanceof JsonObject) {
                    Logging.warn("The GeoJSON contains an object with property '" + (String)stringJsonValueEntry.getKey() + "' whose value has the unsupported type '" + value.getClass().getSimpleName() + "'. That key-value pair is ignored!");
                    continue;
                }
                if (value.getValueType() == JsonValue.ValueType.NULL) continue;
                tags.add(new Tag((String)stringJsonValueEntry.getKey(), value.toString()));
            }
        }
        return tags;
    }

    private static boolean isLineDelimited(InputStream source) {
        source.mark(2);
        try {
            int start = source.read();
            if (30 == start) {
                return true;
            }
            source.reset();
        }
        catch (IOException e) {
            Logging.error(e);
        }
        return false;
    }

    @Override
    protected DataSet doParseDataSet(InputStream source, ProgressMonitor progressMonitor) throws IllegalDataException {
        try (InputStream markSupported = source.markSupported() ? source : new BufferedInputStream(source);){
            this.ds.setUploadPolicy(UploadPolicy.DISCOURAGED);
            if (GeoJSONReader.isLineDelimited(markSupported)) {
                try (BufferedReader reader = new BufferedReader(new InputStreamReader(markSupported, StandardCharsets.UTF_8));){
                    String line;
                    String rs = new String(new byte[]{30}, StandardCharsets.US_ASCII);
                    while ((line = reader.readLine()) != null) {
                        line = Utils.strip(line, rs);
                        JsonParser parser = Json.createParser(new StringReader(line));
                        Throwable throwable = null;
                        try {
                            this.parse(parser);
                        }
                        catch (Throwable throwable2) {
                            throwable = throwable2;
                            throw throwable2;
                        }
                        finally {
                            if (parser == null) continue;
                            GeoJSONReader.$closeResource(throwable, parser);
                        }
                    }
                }
            }
            try (JsonParser parser = Json.createParser(markSupported);){
                this.parse(parser);
            }
            this.mergeEqualMultipolygonWays();
        }
        catch (JsonParsingException | IOException | IllegalArgumentException e) {
            throw new IllegalDataException(e);
        }
        return this.getDataSet();
    }

    private void mergeEqualMultipolygonWays() {
        DuplicateWay test = new DuplicateWay();
        test.startTest(null);
        for (Way w : this.getDataSet().getWays()) {
            test.visit(w);
        }
        test.endTest();
        if (test.getErrors().isEmpty()) {
            return;
        }
        for (TestError e : test.getErrors()) {
            if (e.getPrimitives().size() != 2 || e.isFixable()) continue;
            ArrayList<Way> mpWays = new ArrayList<Way>();
            Way replacement = null;
            for (OsmPrimitive osmPrimitive : e.getPrimitives()) {
                if (osmPrimitive.isTagged() && !osmPrimitive.referrers(Relation.class).findAny().isPresent()) {
                    replacement = (Way)osmPrimitive;
                    continue;
                }
                if (!osmPrimitive.referrers(Relation.class).anyMatch(IRelation::isMultipolygon)) continue;
                mpWays.add((Way)osmPrimitive);
            }
            if (replacement == null && mpWays.size() == 2) {
                replacement = (Way)mpWays.remove(1);
            }
            if (replacement == null || mpWays.size() != 1) continue;
            Way mpWay = (Way)mpWays.get(0);
            for (Relation r : mpWay.referrers(Relation.class).filter(IRelation::isMultipolygon).collect(Collectors.toList())) {
                for (int i = 0; i < r.getMembersCount(); ++i) {
                    if (!r.getMember(i).getMember().equals(mpWay)) continue;
                    r.setMember(i, new RelationMember(r.getRole(i), replacement));
                }
            }
            mpWay.setDeleted(true);
        }
        this.ds.cleanupDeletedPrimitives();
    }

    public static DataSet parseDataSet(InputStream source, ProgressMonitor progressMonitor) throws IllegalDataException {
        return new GeoJSONReader().doParseDataSet(source, progressMonitor);
    }
}

