﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/drs/DrsRequest.h>
#include <aws/drs/Drs_EXPORTS.h>
#include <aws/drs/model/LaunchDisposition.h>
#include <aws/drs/model/LaunchIntoInstanceProperties.h>
#include <aws/drs/model/Licensing.h>
#include <aws/drs/model/TargetInstanceTypeRightSizingMethod.h>

#include <utility>

namespace Aws {
namespace drs {
namespace Model {

/**
 */
class UpdateLaunchConfigurationRequest : public DrsRequest {
 public:
  AWS_DRS_API UpdateLaunchConfigurationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateLaunchConfiguration"; }

  AWS_DRS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Whether we should copy the Private IP of the Source Server to the Recovery
   * Instance.</p>
   */
  inline bool GetCopyPrivateIp() const { return m_copyPrivateIp; }
  inline bool CopyPrivateIpHasBeenSet() const { return m_copyPrivateIpHasBeenSet; }
  inline void SetCopyPrivateIp(bool value) {
    m_copyPrivateIpHasBeenSet = true;
    m_copyPrivateIp = value;
  }
  inline UpdateLaunchConfigurationRequest& WithCopyPrivateIp(bool value) {
    SetCopyPrivateIp(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether we want to copy the tags of the Source Server to the EC2 machine of
   * the Recovery Instance.</p>
   */
  inline bool GetCopyTags() const { return m_copyTags; }
  inline bool CopyTagsHasBeenSet() const { return m_copyTagsHasBeenSet; }
  inline void SetCopyTags(bool value) {
    m_copyTagsHasBeenSet = true;
    m_copyTags = value;
  }
  inline UpdateLaunchConfigurationRequest& WithCopyTags(bool value) {
    SetCopyTags(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The state of the Recovery Instance in EC2 after the recovery operation.</p>
   */
  inline LaunchDisposition GetLaunchDisposition() const { return m_launchDisposition; }
  inline bool LaunchDispositionHasBeenSet() const { return m_launchDispositionHasBeenSet; }
  inline void SetLaunchDisposition(LaunchDisposition value) {
    m_launchDispositionHasBeenSet = true;
    m_launchDisposition = value;
  }
  inline UpdateLaunchConfigurationRequest& WithLaunchDisposition(LaunchDisposition value) {
    SetLaunchDisposition(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Launch into existing instance properties.</p>
   */
  inline const LaunchIntoInstanceProperties& GetLaunchIntoInstanceProperties() const { return m_launchIntoInstanceProperties; }
  inline bool LaunchIntoInstancePropertiesHasBeenSet() const { return m_launchIntoInstancePropertiesHasBeenSet; }
  template <typename LaunchIntoInstancePropertiesT = LaunchIntoInstanceProperties>
  void SetLaunchIntoInstanceProperties(LaunchIntoInstancePropertiesT&& value) {
    m_launchIntoInstancePropertiesHasBeenSet = true;
    m_launchIntoInstanceProperties = std::forward<LaunchIntoInstancePropertiesT>(value);
  }
  template <typename LaunchIntoInstancePropertiesT = LaunchIntoInstanceProperties>
  UpdateLaunchConfigurationRequest& WithLaunchIntoInstanceProperties(LaunchIntoInstancePropertiesT&& value) {
    SetLaunchIntoInstanceProperties(std::forward<LaunchIntoInstancePropertiesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The licensing configuration to be used for this launch configuration.</p>
   */
  inline const Licensing& GetLicensing() const { return m_licensing; }
  inline bool LicensingHasBeenSet() const { return m_licensingHasBeenSet; }
  template <typename LicensingT = Licensing>
  void SetLicensing(LicensingT&& value) {
    m_licensingHasBeenSet = true;
    m_licensing = std::forward<LicensingT>(value);
  }
  template <typename LicensingT = Licensing>
  UpdateLaunchConfigurationRequest& WithLicensing(LicensingT&& value) {
    SetLicensing(std::forward<LicensingT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the launch configuration.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  UpdateLaunchConfigurationRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether we want to enable post-launch actions for the Source Server.</p>
   */
  inline bool GetPostLaunchEnabled() const { return m_postLaunchEnabled; }
  inline bool PostLaunchEnabledHasBeenSet() const { return m_postLaunchEnabledHasBeenSet; }
  inline void SetPostLaunchEnabled(bool value) {
    m_postLaunchEnabledHasBeenSet = true;
    m_postLaunchEnabled = value;
  }
  inline UpdateLaunchConfigurationRequest& WithPostLaunchEnabled(bool value) {
    SetPostLaunchEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the Source Server that we want to retrieve a Launch Configuration
   * for.</p>
   */
  inline const Aws::String& GetSourceServerID() const { return m_sourceServerID; }
  inline bool SourceServerIDHasBeenSet() const { return m_sourceServerIDHasBeenSet; }
  template <typename SourceServerIDT = Aws::String>
  void SetSourceServerID(SourceServerIDT&& value) {
    m_sourceServerIDHasBeenSet = true;
    m_sourceServerID = std::forward<SourceServerIDT>(value);
  }
  template <typename SourceServerIDT = Aws::String>
  UpdateLaunchConfigurationRequest& WithSourceServerID(SourceServerIDT&& value) {
    SetSourceServerID(std::forward<SourceServerIDT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether Elastic Disaster Recovery should try to automatically choose the
   * instance type that best matches the OS, CPU, and RAM of your Source Server.</p>
   */
  inline TargetInstanceTypeRightSizingMethod GetTargetInstanceTypeRightSizingMethod() const {
    return m_targetInstanceTypeRightSizingMethod;
  }
  inline bool TargetInstanceTypeRightSizingMethodHasBeenSet() const { return m_targetInstanceTypeRightSizingMethodHasBeenSet; }
  inline void SetTargetInstanceTypeRightSizingMethod(TargetInstanceTypeRightSizingMethod value) {
    m_targetInstanceTypeRightSizingMethodHasBeenSet = true;
    m_targetInstanceTypeRightSizingMethod = value;
  }
  inline UpdateLaunchConfigurationRequest& WithTargetInstanceTypeRightSizingMethod(TargetInstanceTypeRightSizingMethod value) {
    SetTargetInstanceTypeRightSizingMethod(value);
    return *this;
  }
  ///@}
 private:
  bool m_copyPrivateIp{false};
  bool m_copyPrivateIpHasBeenSet = false;

  bool m_copyTags{false};
  bool m_copyTagsHasBeenSet = false;

  LaunchDisposition m_launchDisposition{LaunchDisposition::NOT_SET};
  bool m_launchDispositionHasBeenSet = false;

  LaunchIntoInstanceProperties m_launchIntoInstanceProperties;
  bool m_launchIntoInstancePropertiesHasBeenSet = false;

  Licensing m_licensing;
  bool m_licensingHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  bool m_postLaunchEnabled{false};
  bool m_postLaunchEnabledHasBeenSet = false;

  Aws::String m_sourceServerID;
  bool m_sourceServerIDHasBeenSet = false;

  TargetInstanceTypeRightSizingMethod m_targetInstanceTypeRightSizingMethod{TargetInstanceTypeRightSizingMethod::NOT_SET};
  bool m_targetInstanceTypeRightSizingMethodHasBeenSet = false;
};

}  // namespace Model
}  // namespace drs
}  // namespace Aws
